<?php
/*
Plugin Name: MasteRRotator – URL Rotator & Tracker
Plugin URI: https://masterrotator.com
Description: Precision URL rotation with tracking. Free version includes sequential/random logic. Pro unlocks advanced analytics and targeting.
Version: 1.0
Author: MasteRRotator Team
Author URI: https://masterrotator.com
Text Domain: masterrotator
*/

if (!defined('ABSPATH')) exit;

// Create tracking table
function url_rotator_create_table() {
    global $wpdb;
    $table = $wpdb->prefix . 'url_rotator_clicks';
    $charset = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table (
        id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
        timestamp DATETIME DEFAULT CURRENT_TIMESTAMP,
        ip_address VARCHAR(45),
        user_agent TEXT,
        target_url TEXT
    ) $charset;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}
register_activation_hook(__FILE__, 'url_rotator_create_table');

// Register settings
add_action('admin_init', function() {
    register_setting('url_rotator_group', 'url_rotator_slug');
    register_setting('url_rotator_group', 'url_rotator_index');
    register_setting('url_rotator_group', 'url_rotator_mode');
    register_setting('url_rotator_group', 'url_rotator_per_user');
    register_setting('url_rotator_group', 'url_rotator_targets', [
        'type' => 'array',
        'sanitize_callback' => function($input) {
            return array_filter(array_map('trim', $input));
        }
    ]);
});

// Admin UI
add_action('admin_menu', function() {
    add_options_page('URL Rotator Pro', 'URL Rotator Pro', 'manage_options', 'url-rotator', 'url_rotator_settings_page');
    add_menu_page('Rotator Clicks', 'Rotator Clicks', 'manage_options', 'url-rotator-clicks', 'url_rotator_clicks_page', 'dashicons-chart-line', 81);
});

function url_rotator_settings_page() {
    ?>
    <div class="wrap">
        <h1>Sequential URL Rotator Pro</h1>
        <form method="post" action="options.php">
            <?php settings_fields('url_rotator_group'); ?>
            <?php do_settings_sections('url_rotator_group'); ?>
            <table class="form-table">
                <tr><th>Slug (e.g. /go)</th>
                    <td><input type="text" name="url_rotator_slug" value="<?php echo esc_attr(get_option('url_rotator_slug', 'go')); ?>" /></td>
                </tr>
                <tr><th>Target URLs (max 10)</th>
                    <td>
                        <?php
                        $raw = get_option('url_rotator_targets');
                        $urls = is_array($raw) ? $raw : array_filter(array_map('trim', explode("\n", $raw)));
                        $urls = array_pad($urls, 10, '');
                        for ($i = 0; $i < 10; $i++) {
                            echo '<input type="text" name="url_rotator_targets[]" value="' . esc_attr($urls[$i]) . '" style="width:100%;margin-bottom:5px;" />';
                        }
                        ?>
                    </td>
                </tr>
                <tr><th>Rotation Mode</th>
                    <td>
                        <select name="url_rotator_mode">
                            <option value="sequential" <?php selected(get_option('url_rotator_mode'), 'sequential'); ?>>Sequential</option>
                            <option value="random" <?php selected(get_option('url_rotator_mode'), 'random'); ?>>Random</option>
                        </select>
                    </td>
                </tr>
                <tr><th>Per-User Rotation</th>
                    <td><input type="checkbox" name="url_rotator_per_user" value="1" <?php checked(get_option('url_rotator_per_user'), 1); ?> /></td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}

// Redirect logic
add_action('init', function() {
    $slug = trim(get_option('url_rotator_slug', 'go'), '/');
    $request = trim($_SERVER['REQUEST_URI'], '/');

    if ($request === $slug) {
        $raw = get_option('url_rotator_targets');
        $targets = is_array($raw) ? $raw : array_filter(array_map('trim', explode("\n", $raw)));
        $targets = array_filter(array_map('trim', $targets));
        $targets = array_slice($targets, 0, 10);

        $mode = get_option('url_rotator_mode', 'sequential');
        $per_user = get_option('url_rotator_per_user', false);
        $index = 0;

        if ($per_user) {
            $cookie = 'url_rotator_index_' . md5($slug);
            $index = isset($_COOKIE[$cookie]) ? (int)$_COOKIE[$cookie] : 0;
        } else {
            $index = (int)get_option('url_rotator_index', 0);
        }

        if (!empty($targets)) {
            $next_url = ($mode === 'random')
                ? $targets[array_rand($targets)]
                : $targets[$index % count($targets)];

            if ($per_user) {
                setcookie($cookie, ($index + 1) % count($targets), time() + 3600 * 24 * 30, "/");
            } else {
                update_option('url_rotator_index', ($index + 1) % count($targets));
            }

            global $wpdb;
            $wpdb->insert($wpdb->prefix . 'url_rotator_clicks', [
                'ip_address' => $_SERVER['REMOTE_ADDR'],
                'user_agent' => $_SERVER['HTTP_USER_AGENT'],
                'target_url' => $next_url
            ]);

            wp_redirect($next_url, 302);
            exit;
        }
    }
});

// Daily summary dashboard
function url_rotator_clicks_page() {
    global $wpdb;
    $table = $wpdb->prefix . 'url_rotator_clicks';

    $results = $wpdb->get_results("
        SELECT 
            DATE(timestamp) as click_date,
            target_url,
            COUNT(*) as total_clicks,
            COUNT(DISTINCT ip_address) as unique_clicks
        FROM $table
        GROUP BY click_date, target_url
        ORDER BY click_date DESC, target_url ASC
        LIMIT 100
    ");

    echo '<div class="wrap"><h1>Rotator Click Summary</h1>';
    echo '<table class="widefat fixed striped"><thead><tr>
        <th>Date</th><th>Target URL</th><th>Total Clicks</th><th>Unique Clicks</th>
    </tr></thead><tbody>';

    foreach ($results as $row) {
        echo '<tr>';
        echo '<td>' . esc_html($row->click_date) . '</td>';
        echo '<td>' . esc_html($row->target_url) . '</td>';
        echo '<td>' . esc_html($row->total_clicks) . '</td>';
        echo '<td>' . esc_html($row->unique_clicks) . '</td>';
        echo '</tr>';
    }

    echo '</tbody></table></div>';
}
